function ball = makeBall(Nx, Ny, Nz, cx, cy, cz, radius)
%MAKEBALL   Create a binary map of filled ball within a 3D grid.
%
% DESCRIPTION:
%       makeBall creates a binary map of a filled ball within a
%       three-dimensional grid (the ball position is denoted by 1's in the
%       matrix with 0's elsewhere). A single pixel is taken as the disc 
%       centre thus the total diameter of the ball will always be an odd
%       number of pixels. 
%
% USAGE:
%       makeBall(Nx, Ny, Nz, cx, cy, cz, radius)
%
% INPUTS:
%       Nx, Ny, Nz      - size of the 3D grid [number of pixels]
%       cx, cy, cz      - centre of the ball [pixel coordinates]
%       radius          - ball radius [number of pixels]
%
% OUTPUTS:
%       ball            - 3D binary map of a filled ball
%
% ABOUT:
%       author          - Bradley Treeby
%       date            - 1st July 2009
%       last update     - 17th July 2009
%       
% This function is part of the k-Wave Toolbox (http://www.k-wave.org)
%
% See also makeCircle, makeDisc, makeSphere 

% define literals
MAGNITUDE = 1;

% create empty matrix
ball = zeros(Nz, Nx, Ny);

% define pixel map
r = makePixelMap(Nx, Ny, Nz, 'Shift', [0 0 0]);

% create disc
ball(r < radius) = MAGNITUDE;

% shift centre
cx = cx - ceil(Nx/2);
cy = cy - ceil(Ny/2);
cz = cz - ceil(Nz/2);
ball = circshift(ball, [cz cx cy]);